//
//  ITMGEngine.h
//  QAVSDK
//


#import <Foundation/Foundation.h>
#import "QAVError.h"
#import "av_type.h"



//#if (TARGET_OS_OSX == 1)
//#define TMG_NO_PTT_SUPPORT
//#endif

typedef NS_ENUM(NSUInteger, ITMG_AUDIO_CATEGORY) {
    ITMG_CATEGORY_AMBIENT = 0,
    ITMG_CATEGORY_PLAYBACK = 1,
};
@class ITMGRoom;
@class ITMGAudioCtrl;
@class ITMGAudioEffectCtrl;
@class ITMGAudioRecordCtrl;
@class ITMGRoomManager;
//#ifndef TMG_NO_PTT_SUPPORT
@class ITMGPTT;
//#endif//TMG_NO_PTT_SUPPORT
@class ITMGFaceTracker;
@class ITMGFaceRenderer;
@class ITMGPoseTracker;

@protocol ITMGDelegate <NSObject>

/*!
 @abstract      sdk事件通知
 @discussion    用于通知TMG事件
 @param         eventType            事件类型
 @param         data  事件数据，是一个字典，具体键值对参考接入文档
 */
- (void)OnEvent:(ITMG_MAIN_EVENT_TYPE)eventType data:(NSDictionary*)data;
@end



@interface ITMGContext : NSObject
/**
 *  房间状态通知delegate
 */
{
    id<ITMGDelegate> _TMGDelegate;
}
@property  (assign, nonatomic)  id<ITMGDelegate> TMGDelegate;

+(ITMGContext*) GetInstance;

/**
 *  设置app信息
 *
 *  @param sdkAppID 设置sdkAppID
 *  @param openID   设置当前用户的openID
 */
-(int)InitEngine:(NSString*)sdkAppID openID:(NSString*)openID;

-(int)Uninit;

-(void)Poll;

-(QAVResult)Pause;
-(QAVResult)Resume;

-(int)SetLogLevel:(ITMG_LOG_LEVEL)levelWrite :(ITMG_LOG_LEVEL)levelPrint;
-(int)SetLogPath:(NSString*)logDir;


/**
 *  获取sdk版本信息
 *
 *  @return 版本号
 */
-(NSString*)GetSDKVersion;

/**
 *  设置app的版本信息
 *
 *  @param appVersion app的版本
 */
-(void)SetAppVersion:(NSString*)appVersion;

/**
 *  设置SDK可以服务的区域
 *
 *  @param region 参见文档，服务的区域简称
 */
-(void)SetRegion:(NSString*)region;

/**
 *  自定义使用的域名
 *
 *  @param chatHost  实时语音使用的域名
 *  @param pttHost  ptt使用的域名
 */

-(void)SetHost:(NSString*)chatHost pttHost:(NSString*)pttHost;
/**
 *  当前是否在音视频房间
 *
 *  @return 在房间返回YES
 */
-(BOOL)IsRoomEntered;

/**
 *  进入音视频房间
 *
 *  @param roomID 房间标识
 *  @param roomType 房间类型 ITMG_ROOM_TYPE
 *  @param authBuffer 加密权限标志
 *
 */
-(int)EnterRoom:(NSString*)roomID roomType:(int)roomType authBuffer:(NSData*)authBuffer;


/**
 *  退出房间
 *  @return 当返回值为AV_OK时，异步执行退房并通过异步回调ExitRoomComplete返回。返回值为非AV_OK时表示执行失败，没有回调。
 */
-(int)ExitRoom;


/**
 *  设置最大混音路数，默认6路，如果需要修改请在SetAppInfo之前调用
 *
 */
-(QAVResult)SetRecvMixStreamCount:(int)count;

-(int)SetRangeAudioMode:(ITMG_RANGE_AUDIO_MODE)gameAudioMode;
-(int)SetRangeAudioTeamID:(int)teamID;


-(int)SetAudioRole:(ITMG_AUDIO_MEMBER_ROLE)role;

/**
 *  扩展参数设置
 *
 */
-(QAVResult)SetAdvanceParams:(NSString*)strKey value:(NSString*)strValue;

-(NSString *)GetAdvanceParams:(NSString*)strKey;

/**
 * 本方法只在观众状态下生效，默认是Ambient，可以支持设置为PlayBack。
 * 具体实现为 修改 kAudioSessionProperty_AudioCategory，具体特性 可参照 Apple官方文档
 */
-(QAVResult)SetDefaultAudienceAudioCategory:(ITMG_AUDIO_CATEGORY)audioCategory;


/**
 *  查询麦克风权限
 *
 */
-(ITMG_RECORD_PERMISSION)CheckMicPermission;

/**
 *  查询mic状态
 *
 */
- (ITMG_CHECK_MIC_STATUS)CheckMic;

-(ITMGRoom*)GetRoom;

-(ITMGAudioCtrl*)GetAudioCtrl;

-(ITMGAudioEffectCtrl*)GetAudioEffectCtrl;

-(ITMGRoomManager*)GetRoomManager;

//#ifndef TMG_NO_PTT_SUPPORT
-(ITMGPTT*)GetPTT;
//#endif//TMG_NO_PTT_SUPPORT


-(QAVResult) StartRealTimeASR;

-(QAVResult) StartRealTimeASR:(NSString *)language;

-(QAVResult) StopRealTimeASR;

-(QAVResult) EnableAgeDectection:(bool) enable;
-(QAVResult) InitAgeDectection: (NSString*)binaryPath paramPath:(NSString*)paramPath;

-(int)execute:(NSString *)mehtodName Param:(NSString *)Param;

-(int)SetLicenseForPlugins:(NSString *)license key:(NSString *)key;

/**
 * 初始化脸部识别
 * @param license 授权证书路径
 * @param secretKey 授权证书的秘钥
 * @return 错误码
 */
// NOCA:MethodNaming(统一风格)
-(int) InitFaceTracker:(NSString *)license secretKey:(NSString *)secretKey;

/**
 * 创建脸部采集器
 * @param dirPath 解析使用的模型文件路径
 * @param fileName 模型文件的文件名
 * @return 返回采集器，使用结束后，需要调用 destroy 释放资源
 */
// NOCA:MethodNaming(统一风格)
-(ITMGFaceTracker *) CreateFaceTracker:(NSString*)dirPath configFile:(NSString*)fileName;

/**
 * 创建姿态采集器
 * @param bodyModePath 身体模型配置文件路径
 * @param bodyModeBinPath 身体模型文件路径
 * @param poseModePath 姿态模型配置文件路径
 * @param poseModeBinPath 姿态模型文件路径
 * @param smootherModePath 平滑模型配置文件路径
 * @param smootherModeBinPath 平滑模型文件路径
 * @return 返回采集器，使用结束后，需要调用 destroy 释放资源
 */
// NOCA:MethodNaming(统一风格)
- (ITMGPoseTracker *)CreatePoseTracker:(NSString *)bodyModePath
                       BodyModeBinPath:(NSString *)bodyModeBinPath
                          PoseModePath:(NSString *)poseModePath
                       PoseModeBinPath:(NSString *)poseModeBinPath
                      SmootherModePath:(NSString *)smootherModePath
                   SmootherModeBinPath:(NSString *)smootherModeBinPath;

/**
 * 创建脸部渲染器
 * @param assetPath 渲染使用的模型文件路径
 * @param fileName 模型文件的文件名
 * @return 返回渲染器，使用结束后，需要调用 destroy 释放资源
 */
// NOCA:MethodNaming(统一风格)
-(ITMGFaceRenderer *) CreateFaceRenderer:(NSString*)assetPath configFile:(NSString*)fileName;

@end

/**
 * 脸部采集器
 */
@interface ITMGFaceTracker : NSObject

/**
 * 销毁采集器
 * @return 错误码
 */
- (int) Destroy;

/**
 * 重置采集器
 * @return  错误码
 */
- (int) Reset;

/**
 * 获取采集器参数
 * @param param 待返回的参数指针，需要外部申请内存
 * @return 错误码
 */
- (int) GetParam:(TMGFaceTrackerParam *)param;
/**
 * 设置采集器参数
 * @param param 需要设置的参数值
 * @return 错误码
 */
- (int) SetParam:(const TMGFaceTrackerParam *)param;

/**
 * 采集脸部信息
 * @param data 包含脸部信息的图像数据
 * @param format 图像格式
 * @param width  图像宽度
 * @param height 图像高度
 * @param stride 图像行宽
 * @param orientation 图像的方向
 * @param faceInfo 返回识别的脸部图像信息,使用完后需要调用releaseTrackedFace释放资源
 * @param faceInfoCount 返回识别到的脸部图像的数量
 * @return 错误码
 */
// NOCA:ParameterCount(包含输入输出)
- (int) TrackFace:(unsigned char *)data
      imageFormat:(ITMG_IMG_FORMAT)format
            width:(int)width
           height:(int)height
           stride:(int)stride
      orientation:(ITMG_IMG_ORIENTATION)orientation
     trackedFaces:(TMGFaceTrackerFaceInfo **)faceInfo
     trackedCount:(int *)faceInfoCount;

/**
 * 释放 通过 trackFace 返回的脸部信息的内存数据
 * @param trackedFaces 待释放的指针数据
 */
- (void) ReleaseTrackedFace:(TMGFaceTrackerFaceInfo *)trackedFaces;

@end;

/**
 * 脸部渲染器
 */
@interface ITMGFaceRenderer : NSObject

/**
 * 销毁渲染器
 * @return 错误码
 */
- (int) Destroy;

/**
 * 渲染脸部信息
 * @param dst 包含待渲染新的输出图像数据
 * @param src 渲染时，使用的画布图像数据，SDK内部会将渲染的脸部信息绘制到该画布上
 * @param format 画布和最后输出的图像信息的格式
 * @param width 画布和最后输出的图像信息的宽
 * @param height 画布和最后输出的图像信息的高
 * @param orientation 画布和最后输出的图像信息的方向
 * @param faceInfo 待绘制的脸部信息
 * @param faceInfoCount 待绘制的脸部信息的数量
 * @return 错误码
 */
// NOCA:ParameterCount(包含输入输出)
- (int) RenderFace:(unsigned char *)dst
               src:(unsigned char *)src
       imageFormat:(ITMG_IMG_FORMAT)format
             width:(int)width
            height:(int)height
       orientation:(ITMG_IMG_ORIENTATION)orientation
             faces:(const TMGFaceTrackerFaceInfo *)faceInfo
         faceCount:(int)faceInfoCount;
@end;

/**
 * 姿态采集器
 */
@interface ITMGPoseTracker : NSObject

/**
 * 销毁采集器
 * @return 错误码
 */
- (int) Destroy;

/**
 * 设置返回姿态角度的渲染引擎平台，因为不用的引擎使用的坐标是不同的，SDK针对不通的平台进行了角度适配转换，方便客户直接使用。具体参数内容如下：
 *  "unity" : Unity平台
 *  "unreal" : Unreal平台
 *  "native" : 不做任何矫正的原始坐标
 *
 *  注意：该参数 区分大小写，默认是 "native"
 * @param engine 引擎平台名
 * @return 错误码
 */
- (int) SetRenderEngine:(NSString*)engine;

/**
 * 采集姿态信息
 * @param data 包含姿态信息的图像数据
 * @param format 图像格式
 * @param width  图像宽度
 * @param height 图像高度
 * @param stride 图像行宽
 * @param orientation 图像的方向
 * @param poseInfo 返回采集的
 * @return 错误码
 */
// NOCA:ParameterCount(包含输入输出)
- (int) TrackPose:(unsigned char *)data
      imageFormat:(ITMG_IMG_FORMAT)format
            width:(int)width
           height:(int)height
           stride:(int)stride
      orientation:(ITMG_IMG_ORIENTATION)orientation
         poseInfo:(TMGPoseTrackerPoseInfo *)poseInfo;
@end

@protocol RoomCustomStreamDataCallback <NSObject>
// NOCA:MethodNaming(统一风格)
-(void) OnRoomCustomStreamData:(unsigned char*)data
                        length:(unsigned int)length
                            // NOCA:runtime/int(设计如此)
                           uin:(unsigned long long)uin
                        // NOCA:runtime/int(设计如此)
                     timestamp:(unsigned long long)timestamp;
@end

@interface ITMGRoom :NSObject

/**
 *  获取质量信息
 *
 *  @return 质量信息tips
 */
-(NSString*)GetQualityTips;

/**
 *  获取当前房间号
 *  @return 当前正在进行中的房间号（退出过程中不算）。
 */
-(NSString*)GetRoomID;

/**
 * 进入房间后，管理员可以调用此方法修改房间的音质类型，会影响房间内的所有已加入和未加入成员
 *  @param roomType 房间类型 ITMG_ROOM_TYPE
 */
-(int)ChangeRoomType:(int)roomType;

/**
 * 进入房间后，可以调用此方法获取当前房间的音质类型；
 * @return roomType 房间类型 ITMG_ROOM_TYPE
 */
-(int)GetRoomType;

// range : if Spatializer is enabled or WorldMode is selected:
//        user can't hear the speaker if the distance between them is larger than the range;
//        by default, range = 0. which means without calling UpdateAudioRecvRange no audio would be available.
-(int) UpdateAudioRecvRange : (int) range;

-(int)UpdateSpatializerRecvRange:(int)range;

// Tell Self's position and rotation information to GME for function: Spatializer && WorldMode
// position and rotate should be under the world coordinate system specified by forward, rightward, upward direction.
// for example: in Unreal(forward->X, rightward->Y, upward->Z); in Unity(forward->Z, rightward->X, upward->Y)
// position: self's position
// axisForward: the forward axis of self's camera rotation
// axisRightward: the rightward axis of self's camera rotation
// axisUpward: the upward axis of self's camera rotation
-(int) UpdateSelfPosition:(int[3])position  axisForward:(float[3])axisForward
                            axisRightward:(float[3])axisRightward axisUpward:(float[3])axisUpward;

-(int) UpdateOtherPosition:(NSString*) openID position:(int[3])position;

-(int)SetServerAudioRoute:(ITMG_SERVER_AUDIO_ROUTE_SEND_TYPE) Sendtype  SendList:(NSArray *)OpenIDForSend  RecvOperateType:(ITMG_SERVER_AUDIO_ROUTE_RECV_TYPE) Recvtype RecvList:(NSArray *)OpenIDForRecv;

-(ITMG_SERVER_AUDIO_ROUTE_SEND_TYPE)GetCurrentSendAudioRoute:(NSMutableArray*) OpenIDForSend;

-(ITMG_SERVER_AUDIO_ROUTE_RECV_TYPE)GetCurrentRecvAudioRoute:(NSMutableArray *)OpenIDForRecv;

/**
* 与某用户连麦
 targetRoomID 目标用户的房间号
 targetOpenID 目标用户的openid
 authBuffer 保留标志位，建议填 NULL
*/

-(int) StartRoomSharing:(NSString *)targetRoomID targetOpenID:(NSString *)targetOpenID authBuffer:(NSData*)authBuffer;



/**
* 停止与某用户连麦
*/

-(int) StopRoomSharing;

/**
  * 快速切换房间
  * @param roomID 新房间roomid
  * @param authBuffer 加密权限标志
 */
-(int) SwitchRoom:(NSString *)roomID  authBuffer:(NSData*)authBuffer;

-(int) SendCustomData:(NSData *)data repeatCout:(int)repeatCout;

-(int) StopSendCustomData;

-(int)StartChorusWithOpenID:(NSString*)openID;

-(int)StopChorus;

-(int)StartChorusVocalAccompaniment:(NSString*)openID;

-(int)StopChorusVocalAccompaniment;

// NOCA:MethodNaming(统一风格)
-(int) SendCustomStreamData:(NSData *)customSteamData;

// NOCA:MethodNaming(统一风格)
- (int) SetCustomStreamDataCallback:(id<RoomCustomStreamDataCallback>)callback;

@end

@interface ITMGRoomManager :NSObject

/**
* 打开/关闭某人的麦克风和上行
*/

-(QAVResult)EnableMic:(BOOL)enable Receiver:(NSString *)receiverID;

/**
* 打开/关闭某人的音频上行开关
*/

-(QAVResult)EnableSpeaker:(BOOL)enable Receiver:(NSString *)receiverID;

/**
* 打开/关闭某人的音频上行开关
*/

-(QAVResult)EnableAudioCaptureDevice:(BOOL)enabled Receiver:(NSString *)receiverID;

/**
* 打开/关闭某人的音频上行开关
*/

-(QAVResult)EnableAudioPlayDevice:(BOOL)enabled Receiver:(NSString *)receiverID;

/**
 * 打开/关闭某人的音频上行开关
 */

-(QAVResult)EnableAudioSend:(BOOL)enable Receiver:(NSString *)receiverID;

/**
 * 打开/关闭某人的音频下行
 */

-(QAVResult)EnableAudioRecv:(BOOL)enable Receiver:(NSString *)receiverID;

/**
//获取某人麦克风状态
 */
-(QAVResult)GetMicState:(NSString *)receiverID;

/**
//获取某人扬声器状态
*/

-(QAVResult)GetSpeakerState:(NSString *)receiverID;

/**
//禁止某人操作麦克风扬声器
 
  */
-(QAVResult)ForbidUserOperation:(BOOL)enable Receiver:(NSString *)receiverID;

@end




@interface ITMGAudioCtrl : NSObject

// 推荐使用此方法开关麦克风。EnalbeMic(value) = EnableAudioCaptureDevice(value) + EnableAudioSend(value)
-(QAVResult)EnableMic:(BOOL)enable;
// 只是快捷方式。[0 is off; 1 is on] GetMicState  = IsAudioSendEnabled  && IsAudioCaptureDeviceEnabled
-(int) GetMicState;

// 推荐使用此方法开关扬声器。EnableSpeaker(value) = EnableAudioPlayDevice(value) + EnableAudioRecv(value)
-(QAVResult)EnableSpeaker:(BOOL)enable;
// 只是快捷方式。[0 is off; 1 is on] GetSpeakerState  = IsAudioRecvEnabled  && IsAudioPlayDeviceEnabled
-(int) GetSpeakerState;

/**
 *  @param count 进房后设置实际混音路数，必须小于SetRecvMixStreamCount中的参数
 *  @return GME通用返回值，AV_OK表示成功
 */
-(QAVResult)SetAudioMixCount:(int)count;
/**
 * 开启采集和播放设备。默认情况下，GME不会打开设备。
 * 注意：只能在进房后调用此API，退房会自动关闭设备
 * 注意：在移动端，打开采集设备通常会伴随权限申请，音量类型调整等操作。
 *
 * 调用场景举例：
 * 1、当用户界面点击打开/关闭麦克风/扬声器按钮时，建议：
 *     Option 1: 对于大部分的游戏类App，总是应该同时调用EnableAudioCaptureDevice/EnableAudioSend 和 EnableAudioPlayDevice/EnableAudioRecv
 *     Option 2: 只有特定的社交类App，需要在进房后一次性调用EnableAudioCapture/PlayDevice(true)，后续只使用EnableAudioSend/Recv进行音频流控制
 * 2、如目的是互斥（释放录音权限给其他模块使用），建议使用PauseAudio/ResumeAudio。
 */
-(QAVResult)EnableAudioCaptureDevice:(BOOL)enabled;
-(QAVResult)EnableAudioPlayDevice:(BOOL)enabled;

-(BOOL)IsAudioCaptureDeviceEnabled;
-(BOOL)IsAudioPlayDeviceEnabled;

-(int)SetSpeakerVolumeByOpenID:(NSString*)openId volume:(int)volume;
-(int)GetSpeakerVolumeByOpenID:(NSString*)openId;

/**
 * 打开/关闭音频上行，如果采集设备已经打开，那么会发送采集到的音频数据。如果采集设备没有打开，那么仍旧无声。参见EnableAudioCaptureDevice
 * ExitRoom will automatically call EnableAudioSend(false)
 */
-(QAVResult)EnableAudioSend:(BOOL)enable;
/**
 * 打开/关闭音频下行，如果播放设备已经打开，那么会接收并播放房间里其他人的音频数据。如果播放设备没有打开，那么仍旧无声。参见EnableAudioPlayDevice
 * ExitRoom will automatically call EnableAudioRecv(false)
 */
-(QAVResult)EnableAudioRecv:(BOOL)enable;

-(BOOL)IsAudioSendEnabled;
-(BOOL)IsAudioRecvEnabled;

/**
 *  检测 iPhone 静音键状态
 *
 *  @return 该次请求的结果，不是状态，状态需要在回调中查看。
 */
- (QAVResult)CheckDeviceMuteState;


/**
 *  获取麦克风实时音量，可以用来绘制波形图
 *
 *  @return 音量值，取值范围[0,100]
 */
-(int)GetMicLevel;


/**
 *  设置麦克风软件音量
 *
 *  @param volume 音量值，取值范围[0,200]
 */
-(QAVResult)SetMicVolume:(int) volume;

/**
 *  获取麦克风软件音量，默认100
 */
-(int) GetMicVolume;

/**
 *  获取扬声器实时音量，可以用来绘制波形图
 *
 *  @return 音量值，取值范围[0,100]
 */
-(int)GetSpeakerLevel;

/**
 *  设置扬声器软件音量
 *
 *  @param volume 音量值，取值范围[0,200]
 */
-(QAVResult)SetSpeakerVolume:(int)volume;
/**
 *  获取扬声器软件音量
 *
 *  @return 音量值，取值范围[0,200]
 */
-(int)GetSpeakerVolume;

/**
 *  打开或关闭声音回环，打开后演示器会播放自己的声音
 *
 *  @param enable 是否打开
 *
 *  @return 参考QAVResult
 */
-(QAVResult)EnableLoopBack:(BOOL)enable;

/**
 *  采集系统声音并混音发送，房间内可以听到系统播放的声音
 *
 *  @param enable 是否打开
 *
 *  @return 参考QAVResult
 */
-(QAVResult)EnableMixSystemAudioToSend:(BOOL)enable;

/**
 *  获取自己上行的实时音量
 *
 *  @return 音量值，取值范围[0,100]
 */
- (int)GetSendStreamLevel;

/**
 *  获取房间内其他成员下行流的实时音量
 *
 *  @param openID 房间内其他成员的openid
 *  @return 音量值，取值范围[0,100]
 */
- (int)GetRecvStreamLevel:(NSString*)openID;

/**
 * 音频黑名单
 */
-(QAVResult)AddAudioBlackList:(NSString*)openID;

-(QAVResult)RemoveAudioBlackList:(NSString*)openID;
-(BOOL)IsOpenIdInAudioBlackList:(NSString*)openID;
/*
 * Init Spatializer, If you want to use EnableSpatializer, InitSpatializer should be called on both speaker and listener's client side
*/
-(QAVResult)InitSpatializer:(NSString*) modelPath;

/*
// Enable or disable the spatial audio
//[in]bEnable	--> Enable or disable the spatial audio
//[in]applyTeam	--> indicates whether to use spatial audio in the same team or not
// note: InitSpatializer should be called on both speaker and listener's client side
 */
-(QAVResult)EnableSpatializer:(BOOL)bEnable applyTeam:(BOOL)applyTeam;

/*
// Get the current spatial audio state
 */
-(BOOL)IsEnableSpatializer;

/**
 *  跟踪房间成员音量变化
 *  @param interval 通知间隔
 *  @return 参考QAVResult
 */

-(QAVResult)TrackingVolume:(float)interval;

/**
 *  停止跟踪房间成员音量变化
*  @return 参考QAVResult
 */
-(QAVResult)StopTrackingVolume;


/**
 *  将Team内成员 加入3D效果
 *   @return 参考QAVResult
 */
-(QAVResult)AddSameTeamSpatializer:(NSString*) openID;

/**
 *  将Team内成员 取消3D效果（默认为取消状态）
 *  @return 参考QAVResult
 */
-(QAVResult)RemoveSameTeamSpatializer:(NSString*) openID;

/**
 *  将房间内成员添加到3D效果黑名单中（处于黑名单中的用户的声音没有3D效果）
 *  @return 参考QAVResult
 */
-(QAVResult)AddSpatializerBlacklist:(NSString*) openID;

/**
 *  将房间内成员从3D效果黑名单中移除（处于黑名单中的用户的声音没有3D效果）
 *  @return 参考QAVResult
 */
-(QAVResult)RemoveSpatializerBlacklist:(NSString*) openID;

/**
 *  清空3D效果黑名单（处于黑名单中的用户的声音没有3D效果）
 *  @return 参考QAVResult
 */
-(QAVResult)ClearSpatializerBlacklist;


@end

@interface ITMGAudioEffectCtrl:NSObject
/**
 *  开始伴奏
 *
 *  @param filePath  伴奏文件
 *  @param loopBack  是否其他人也可以听到，false为其他人听不到，true是其他人也能听到。
 *  @param loopCount 循环的次数，-1为无限循环。
*  @return 参考QAVResult
 */
-(QAVResult)StartAccompany:(NSString*)filePath loopBack:(BOOL)loopBack loopCount:(int)loopCount;

-(QAVResult)StartRecordForHardwareDelayTest;

-(QAVResult)StopRecordForHardwareDelayTest;

-(QAVResult)StartPreviewDelayTest;

-(QAVResult)StopPreviewDelayTest;

-(int)SetHardWareDelay:(int)delayinMS;

-(int)GetHardWareDelay;

/**
 *  升降key功能
 *  @nKey 升降key 建议范围-4到4
*/
-(QAVResult)SetAccompanyKey:(int)nKey;

/**
 *  停止伴奏
 *
 *  @param duckerTime 填0
 *
 *  @return 参考QAVResult
 */
-(QAVResult)StopAccompany:(int)duckerTime;

/**
 *  伴奏是否结束
 *
 *  @return true表示已经结束
 */
-(bool)IsAccompanyPlayEnd;
/**
 *  暂停伴奏
 *
 *  @return 参考QAVResult
 */
-(QAVResult)PauseAccompany;
/**
 *  恢复伴奏
 *
 *  @return 参考QAVResult
 */
-(QAVResult)ResumeAccompany;

/**
 *  设置是否自己可以听到伴奏
 *
 *  @param enable false为自己听不到，true是自己能听到
 *
 *  @return 参考QAVResult
 */
-(QAVResult)EnableAccompanyPlay:(BOOL)enable;

/**
 *  设置是否其他人也可以听到伴奏。
 *
 *  @param enable alse为其他人听不到，true是其他人也能听到
 *
 *  @return 参考QAVResult
 */
-(QAVResult)EnableAccompanyLoopBack:(BOOL)enable;

/**
 *  设置伴奏音量，为线性音量
 *
 *  @param vol 默认值为100，大于100增益，小于100减益
 *
 *  @return 参考QAVResult
 */
-(QAVResult)SetAccompanyVolume:(int)vol;
/**
 *  获取伴奏音量
 *
 *  @return 音量值
 */
-(int)GetAccompanyVolume;


/**
 *  获取自己的伴奏总时长
 *
 *  @return 时长
 */
-(int)GetAccompanyFileTotalTimeByMs;
/**
 *  获取自己的伴奏播放进度
 *
 *  @return 当前进度
 */
-(int)GetAccompanyFileCurrentPlayedTimeByMs;

/**
 *  获取指定用户的伴奏总时长，只对 roomtype 2和3有效
 *
 *  @return 时长
 */
-(int)GetAccompanyFileTotalTimeByMs:(NSString*)openId;
/**
 *  获取指定用户的伴奏播放进度 ，只对 roomtype 2和3有效
 *
 *  @return 当前进度
 */
-(int)GetAccompanyFileCurrentPlayedTimeByMs:(NSString*)openId;

/**
 *  设置当前播放位置
 *
 *  @param time 播放位置,毫秒为单位
 *
 *  @return  参考QAVResult
 */
-(QAVResult)SetAccompanyFileCurrentPlayedTimeByMs:(uint) time;


/**
 *  获取音效音量
 *
 *  @return 音效音量，默认值为100，大于100增益，小于100减益
 */
-(int)GetEffectsVolume;
/**
 *  设置音效音量
 *
 *  @param volume 音量值，默认值为100，大于100增益，小于100减益
 *
 *  @return 参考QAVResult
 */
-(QAVResult)SetEffectsVolume:(int)volume;

/**
 *  获取指定soundId的音效音量
 *
 *  @return 音效音量，默认值为100，大于100增益，小于100减益
 */
-(int)GetEffectVolume:(int)soundId;

/**
 *  设置指定soundId的音效音量
 *
 *  @param volume 音量值，默认值为100，大于100增益，小于100减益
 *
 *  @return 参考QAVResult
 */
-(QAVResult)SetEffectVolume:(int)soundId volume:(int)volume;

/**
 *  播放音效文件
 *
 *  @param soundId  设置当前文件的soundId，后续通过soundId来控制该音效
 *  @param filePath 音效文件路径
 *  @param loop     是否循环播放
 *
 *  @return 参考QAVResult
 */
-(QAVResult)PlayEffect:(int)soundId filePath:(NSString*)filePath loop:(BOOL)loop;
/**
 *  暂停音效播放
 *
 *  @param soundId 音效soundId
 *
 *  @return 参考QAVResult
 */
-(QAVResult)PauseEffect:(int)soundId;

/**
 *  暂停所有音效播放
 *
 *  @return 参考QAVResult
 */
-(QAVResult)PauseAllEffects;

/**
 *  恢复单个音效播放
 *
 *  @param soundId 音效soundId
 *
 *  @return 参考QAVResult
 */
-(QAVResult)ResumeEffect:(int)soundId;
/**
 *  恢复所有音效播放
 *
 *  @return 参考QAVResult
 */
-(QAVResult)ResumeAllEffects;

/**
 *  停止某个音效播放
 *
 *  @param soundId  音效soundId
 *
 *  @return 参考QAVResult
 */
-(QAVResult)StopEffect:(int)soundId;
/**
 *  停止所有音效播放
 *
 *  @return 参考QAVResult
 */
-(QAVResult)StopAllEffects;

/**
 *  设置音效是否可以混流到发送流中
 *
 *  @param soundId  音效soundId
 *  @param enable   是否可以混流到发送流中
 *  @return 参考QAVResult
 */
-(QAVResult)EnableEffectSend:(int)soundId enable:(bool)enable;

/**
 *  设置音效当前的播放位置
 *
 *  @param soundId  音效soundId
 *  @param time     播放位置,毫秒为单位
 *  @return 参考QAVResult
 */
-(QAVResult)SetEffectFileCurrentPlayedTimeByMs:(int)soundId time:(uint)time;
-(int)GetEffectFileCurrentPlayedTimeByMs:(int)soundId;

-(QAVResult)StartRecord:(NSString*)filePath sampleRate:(int)sampleRate channels:(int)channels recordLocalMic:(bool) recordLocalMic recordRemote:(bool)recordRemote recordAccompany:(bool)recordAccompany;
-(QAVResult)StopRecord;
-(QAVResult)PauseRecord;
-(QAVResult)ResumeRecord;
-(QAVResult)EnableRecordLocalMic:(bool)enable;
-(QAVResult)EnableRecordAccompany:(bool)enable;
-(QAVResult)EnableRecordRemote:(bool)enable;

/**
 *  变声
 *
 *  @return 参考QAVResult
 */
-(QAVResult)SetVoiceType:(ITMG_VOICE_TYPE) type;

/**
 *  卡拉ok音效接口
 */
-(QAVResult)SetKaraokeType:(ITMG_KARAOKE_TYPE) type;

/**
 *  卡拉ok音效接口
 */
-(QAVResult)SetKaraokeType:(ITMG_VOICE_TYPE_EQUALIZER*)equalizer reverb:(ITMG_VOICE_TYPE_REVERB*)reverb;

-(QAVResult)InitVoiceChanger:(NSString *)dataPath;
-(QAVResult)SetVoiceChangerName:(NSString *)name;
-(QAVResult)FetchVoiceChangerList;
-(NSString *)GetVoiceChangerParams;
-(float)GetVoiceChangerParamValue:(NSString *)paramName;
-(QAVResult)SetVoiceChangerParam:(NSString *)paramName andValue:(float)value;
@end

//#ifndef TMG_NO_PTT_SUPPORT

@interface ITMGPTT : NSObject

-(QAVResult)SetMaxMessageLength:(int)msTime;
/**
 *  设置语音的原始语言，在开启语音审核时，需要设置此参数 ，默认是”cmn-Hans-CN“。需要在UploadRecordedFile前调用。
 *
 *  @param sourceLanguage  语音的原始语言
 *
 *  @return 参考QAVResult
 */
- (QAVResult)SetPTTSourceLanguage:(NSString *)sourceLanguage;

-(int)StartRecordingWithStreamingRecognition:(NSString *)filePath;
-(int)StartRecordingWithStreamingRecognition:(NSString *)filePath language:(NSString*)speechLanguage;
-(int)StartRecordingWithStreamingRecognition:(NSString *)filePath language:(NSString*)speechLanguage translatelanguage:(NSString*)translatelanguage;

-(int)StartRecording:(NSString*)filePath;
-(QAVResult)StopRecording;
-(QAVResult)CancelRecording;

-(int)PauseRecording;
-(int)ResumeRecording;

-(int) GetMicLevel;
-(QAVResult)SetMicVolume:(int) volume;
-(int) GetMicVolume;

-(int) GetSpeakerLevel;
-(QAVResult)SetSpeakerVolume:(int)volume;
-(int)GetSpeakerVolume;

-(QAVResult)ApplyPTTAuthbuffer:(NSData *)authBuffer;
-(int)UploadRecordedFile:(NSString*)filePath;
-(int)DownloadRecordedFile:(NSString*)fileId filePath:(NSString*)filePath;
-(int)PlayRecordedFile:(NSString*)filePath;
-(int)PlayRecordedFile:(NSString*)filePath VoiceType:(ITMG_VOICE_TYPE) type;
-(int)StopPlayFile;


-(int)SpeechToText:(NSString*)fileID;
-(int)SpeechToText:(NSString*)fileID language:(NSString*)speechLanguage;
-(int)SpeechToText:(NSString*)fileID language:(NSString*)speechLanguage  translateLanguage:(NSString*)translateLanguage;


-(int)GetFileSize:(NSString*)filePath;

-(int)GetVoiceFileDuration:(NSString*)filePath;
-(int)TranslateText:(NSString*)text sourceLanguage:(NSString*)sourceLanguage translateLanguage:(NSString*)translateLanguage;
-(int)TextToSpeech:(NSString*)text voiceName:(NSString*)voiceName languageCode:(NSString*)languageCode speakingRate:(float)speakingRate;
@end
//#endif//TMG_NO_PTT_SUPPORT
